
// DiscFree.java applet gives a graphical view of DISCFREE C output

import java.applet.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;
import java.net.*;

// Lars Appel, 15-May-00

public class DiscFree extends Applet {

  // store parsed discfree data in a vector of DevRec tuples
  
  Vector v;

  class DevRec {
    int ldev;
    String volSet;
    String member;
    int discSize;
    int permUsed;
    int permFree;
  }

  // fetch collects max disc size for scaling the graph later
  
  int maxDiscSize = 0;

  // data source can be local file (for testing) or URL (eg CGI)
  
  String dataSource;

  // we also have a refresh button above the main drawing area

  Button refreshButton;
  
  Canvas mainArea;
  

  /**
   * Parse applet parameters and fetch/parse initial discfree info
   */
  public void init() {

    String method = getParameter("dataMethod");
    String source = getParameter("dataSource");

    if (method != null && method.equals("http")) {
      URL webPage = getDocumentBase();
      dataSource =
        "http://" + webPage.getHost() + ":" + webPage.getPort()
        + ((source != null) ? source : "/cgi-bin/discfree");
    } else {
      dataSource = (source !=null) ? source : "discfree.txt";
    }

    fetchData(dataSource);

    setLayout(new BorderLayout());
    refreshButton = new Button("Refresh");
    add(refreshButton, "South");

    refreshButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        fetchData(dataSource);
        mainArea.repaint();
      }
    });

    mainArea = new Canvas() {
      public void paint(Graphics g) {
        paintMainArea(g);
      }
    };
    add(mainArea, "Center");
  }


  /**
   * Fetch discfree info from local file or URL and parse it
   */
  private void fetchData(String fileOrURL) {
    try {

      BufferedReader r;
      
      if (fileOrURL.startsWith("http:")) {
        r = new BufferedReader(
          new InputStreamReader(new URL(fileOrURL).openStream())
        );
      } else {
        r = new BufferedReader(new FileReader(fileOrURL));
      }

      v = new Vector();
      
      DevRec d = null;
      
      for (;;) {
        String s = r.readLine();
        if (s == null) break;
        if (s.startsWith("LDEV")) {
          d = new DevRec();
          v.addElement(d);
          StringTokenizer t = new StringTokenizer(s, " :-()");
          t.nextToken();
          d.ldev = Integer.parseInt(t.nextToken());
          d.volSet = t.nextToken();
          d.member = t.nextToken();
        } else
        if (s.startsWith(" Device")) {
          StringTokenizer t = new StringTokenizer(s, " |()");
          t.nextToken();
          d.discSize = (Integer.parseInt(t.nextToken()) + 2048) / 4096;
          if (d.discSize > maxDiscSize) {
            maxDiscSize = d.discSize;
          }
        } else
        if (s.startsWith(" Permanent")) {
          StringTokenizer t = new StringTokenizer(s, " |()");
          t.nextToken();
          t.nextToken();
          t.nextToken();
          d.permUsed = (Integer.parseInt(t.nextToken()) + 2048) / 4096;
          t.nextToken();
          d.permFree = (Integer.parseInt(t.nextToken()) + 2048) / 4096;
        } else
        if (s.startsWith("TOTALS")) {
          break;
        }
      }
      
      r.close();

    }
    catch (IOException e) {
      System.out.println("oops, failed to fetch discfree data");
    }
  }


  /**
   * Paint or repaint applet area with stacked bars and label text
   */
  private void paintMainArea(Graphics g) {

    int fontHeight = g.getFontMetrics().getHeight();

    final int areaLeft = 10;
    final int areaTop = 0;
    final int areaWidth = getSize().width - 20;
    final int areaHeight = getSize().height - 10;
    
    final int sliceHeight = areaHeight / v.size();
    
    final int sliceUpper = 1 * fontHeight;
    final int sliceGraph = 2 * fontHeight;
    final int sliceLower = 3 * fontHeight;
    
    final int hBox = 10;
    final int hBar = 7;
    final int dxBox = 2;
    final int dzBox = 7;
    final int dzBar = 4;

    g.clearRect(areaLeft, areaTop, areaWidth, areaHeight);
    
    for (int k=0; k < v.size(); k++) {

      DevRec d = (DevRec) v.elementAt(k);

      drawBox(
        g, Color.gray,
        areaLeft,
        areaTop + k * sliceHeight + sliceGraph - 3,
        d.discSize * areaWidth / maxDiscSize + dxBox,
        hBox, dzBox
      );
      drawBar(
        g, Color.blue,
        areaLeft,
        areaTop + k * sliceHeight + sliceGraph - 3,
        d.permUsed * areaWidth / maxDiscSize,
        hBar, dzBar
      );
      drawBar(
        g, Color.green,
        areaLeft + d.permUsed * areaWidth / maxDiscSize,
        areaTop + k * sliceHeight + sliceGraph - 3,
        d.permFree * areaWidth / maxDiscSize,
        hBar, dzBar
      );

      g.setColor(Color.black);

      g.drawString(
        "LDEV " + d.ldev + " (" + d.volSet + ":" + d.member + ")",
        areaLeft, areaTop + k * sliceHeight + sliceUpper
      );
      g.drawString(
        d.permUsed + " MB used, "
        + d.permFree + " MB free, "
        + d.discSize + " MB size ",
        areaLeft, areaTop + k * sliceHeight + sliceLower
      );
    }
  }


  /**
   * Draw a 3D box with given coordinates, dimensions and color
   */
  private void drawBox(
    Graphics g, Color c, int x, int y, int w, int h, int dz
  )
  {
    int[] xCoords = new int[] {
      x, x+w, x+w-dz, x-dz, x-dz, x
    };
    
    int[] yCoords = new int[] {
      y, y, y+dz, y+dz, y+dz-h, y-h
    };

    g.setColor(c.brighter());
    g.fillRect(x, y-h, w, h);
    
    g.setColor(c);
    g.fillPolygon(xCoords, yCoords, 6);
  }


  /**
   * Draw a 3D bar with given coordinates, dimensions and color
   */
  private void drawBar(
    Graphics g, Color c, int x, int y, int w, int h, int dz
  )
  {
    int[] xCoords = new int[] {
      x, x+w, x+w, x+w-dz, x+w-dz, x-dz
    };
    
    int[] yCoords = new int[] {
      y-h, y-h, y, y+dz, y+dz-h, y+dz-h
    };
    
    g.setColor(c.darker());
    g.fillPolygon(xCoords, yCoords, 6);

    g.setColor(c);
    g.fillRect(x-dz, y+dz-h, w, h);
  }

}

